/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.util;

import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.common.AttributesBuilder;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Context;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.function.Consumer;
import org.apache.hadoop.fs.ByteBufferReadable;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.hbase.io.hfile.trace.HFileContextAttributesBuilderConsumer;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.trace.HBaseSemanticAttributes;
import org.apache.hadoop.io.IOUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class BlockIOUtils {
    private static final Logger LOG = LoggerFactory.getLogger(BlockIOUtils.class);
    private static Method byteBufferPositionedReadMethod;

    private BlockIOUtils() {
    }

    private static void initByteBufferPositionReadableMethod() {
        try {
            byteBufferPositionedReadMethod = FSDataInputStream.class.getMethod("read", Long.TYPE, ByteBuffer.class);
        }
        catch (NoSuchMethodException e) {
            LOG.debug("Unable to find positioned bytebuffer read API of FSDataInputStream. preadWithExtra() will use a temporary on-heap byte array.");
        }
    }

    public static boolean isByteBufferReadable(FSDataInputStream is) {
        InputStream cur = is.getWrappedStream();
        while (cur instanceof FSDataInputStream) {
            cur = ((FSDataInputStream)cur).getWrappedStream();
        }
        return cur instanceof ByteBufferReadable;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void readFully(ByteBuff buf, FSDataInputStream dis, int length) throws IOException {
        Span span = Span.current();
        AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
        if (!BlockIOUtils.isByteBufferReadable(dis)) {
            byte[] heapBuf = new byte[length];
            IOUtils.readFully((InputStream)dis, (byte[])heapBuf, (int)0, (int)length);
            BlockIOUtils.annotateHeapBytesRead(attributesBuilder, length);
            span.addEvent("BlockIOUtils.readFully", attributesBuilder.build());
            BlockIOUtils.copyToByteBuff(heapBuf, 0, length, buf);
            return;
        }
        int directBytesRead = 0;
        int heapBytesRead = 0;
        ByteBuffer[] buffers = buf.nioByteBuffers();
        int remain = length;
        int idx = 0;
        ByteBuffer cur = buffers[idx];
        try {
            while (remain > 0) {
                while (!cur.hasRemaining()) {
                    if (++idx >= buffers.length) {
                        throw new IOException("Not enough ByteBuffers to read the reminding " + remain + " bytes");
                    }
                    cur = buffers[idx];
                }
                cur.limit(cur.position() + Math.min(remain, cur.remaining()));
                int bytesRead = dis.read(cur);
                if (bytesRead < 0) {
                    throw new IOException("Premature EOF from inputStream, but still need " + remain + " bytes");
                }
                remain -= bytesRead;
                if (cur.isDirect()) {
                    directBytesRead += bytesRead;
                    continue;
                }
                heapBytesRead += bytesRead;
            }
        }
        finally {
            BlockIOUtils.annotateBytesRead(attributesBuilder, directBytesRead, heapBytesRead);
            span.addEvent("BlockIOUtils.readFully", attributesBuilder.build());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void readFullyWithHeapBuffer(InputStream in, ByteBuff out, int length) throws IOException {
        if (length < 0) {
            throw new IllegalArgumentException("Length must not be negative: " + length);
        }
        int heapBytesRead = 0;
        int remain = length;
        byte[] buffer = new byte[1024];
        try {
            while (remain > 0) {
                int count = in.read(buffer, 0, Math.min(remain, buffer.length));
                if (count < 0) {
                    throw new IOException("Premature EOF from inputStream, but still need " + remain + " bytes");
                }
                out.put(buffer, 0, count);
                remain -= count;
                heapBytesRead += count;
            }
        }
        finally {
            Span span = Span.current();
            AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
            BlockIOUtils.annotateHeapBytesRead(attributesBuilder, heapBytesRead);
            span.addEvent("BlockIOUtils.readFullyWithHeapBuffer", attributesBuilder.build());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean readWithExtraOnHeap(InputStream in, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int heapBytesRead = 0;
        int bytesRemaining = necessaryLen + extraLen;
        try {
            while (bytesRemaining > 0) {
                int ret = in.read(buf, bufOffset, bytesRemaining);
                if (ret < 0) {
                    if (bytesRemaining <= extraLen) {
                        break;
                    }
                    throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + (necessaryLen + extraLen - bytesRemaining));
                }
                bufOffset += ret;
                bytesRemaining -= ret;
                heapBytesRead += ret;
            }
        }
        finally {
            Span span = Span.current();
            AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
            BlockIOUtils.annotateHeapBytesRead(attributesBuilder, heapBytesRead);
            span.addEvent("BlockIOUtils.readWithExtra", attributesBuilder.build());
        }
        return bytesRemaining <= 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean readWithExtra(ByteBuff buf, FSDataInputStream dis, int necessaryLen, int extraLen) throws IOException {
        if (!BlockIOUtils.isByteBufferReadable(dis)) {
            byte[] heapBuf = new byte[necessaryLen + extraLen];
            boolean ret = BlockIOUtils.readWithExtraOnHeap((InputStream)dis, heapBuf, 0, necessaryLen, extraLen);
            BlockIOUtils.copyToByteBuff(heapBuf, 0, heapBuf.length, buf);
            return ret;
        }
        int directBytesRead = 0;
        int heapBytesRead = 0;
        ByteBuffer[] buffers = buf.nioByteBuffers();
        int bytesRead = 0;
        int remain = necessaryLen + extraLen;
        int idx = 0;
        ByteBuffer cur = buffers[idx];
        try {
            while (bytesRead < necessaryLen) {
                while (!cur.hasRemaining()) {
                    if (++idx >= buffers.length) {
                        throw new IOException("Not enough ByteBuffers to read the reminding " + remain + "bytes");
                    }
                    cur = buffers[idx];
                }
                cur.limit(cur.position() + Math.min(remain, cur.remaining()));
                int ret = dis.read(cur);
                if (ret < 0) {
                    throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + bytesRead);
                }
                bytesRead += ret;
                remain -= ret;
                if (cur.isDirect()) {
                    directBytesRead += ret;
                    continue;
                }
                heapBytesRead += ret;
            }
        }
        finally {
            Span span = Span.current();
            AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
            BlockIOUtils.annotateBytesRead(attributesBuilder, directBytesRead, heapBytesRead);
            span.addEvent("BlockIOUtils.readWithExtra", attributesBuilder.build());
        }
        return extraLen > 0 && bytesRead == necessaryLen + extraLen;
    }

    public static boolean preadWithExtra(ByteBuff buff, FSDataInputStream dis, long position, int necessaryLen, int extraLen) throws IOException {
        return BlockIOUtils.preadWithExtra(buff, dis, position, necessaryLen, extraLen, false);
    }

    public static boolean preadWithExtra(ByteBuff buff, FSDataInputStream dis, long position, int necessaryLen, int extraLen, boolean readAllBytes) throws IOException {
        boolean preadbytebuffer = dis.hasCapability("in:preadbytebuffer");
        if (preadbytebuffer) {
            return BlockIOUtils.preadWithExtraDirectly(buff, dis, position, necessaryLen, extraLen, readAllBytes);
        }
        return BlockIOUtils.preadWithExtraOnHeap(buff, dis, position, necessaryLen, extraLen, readAllBytes);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean preadWithExtraOnHeap(ByteBuff buff, FSDataInputStream dis, long position, int necessaryLen, int extraLen, boolean readAllBytes) throws IOException {
        int remain = necessaryLen + extraLen;
        byte[] buf = new byte[remain];
        int bytesRead = 0;
        int lengthMustRead = readAllBytes ? remain : necessaryLen;
        try {
            while (bytesRead < lengthMustRead) {
                int ret = dis.read(position + (long)bytesRead, buf, bytesRead, remain);
                if (ret < 0) {
                    throw new IOException("Premature EOF from inputStream (positional read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + bytesRead);
                }
                bytesRead += ret;
                remain -= ret;
            }
        }
        finally {
            Span span = Span.current();
            AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
            BlockIOUtils.annotateHeapBytesRead(attributesBuilder, bytesRead);
            span.addEvent("BlockIOUtils.preadWithExtra", attributesBuilder.build());
        }
        BlockIOUtils.copyToByteBuff(buf, 0, bytesRead, buff);
        return extraLen > 0 && bytesRead == necessaryLen + extraLen;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean preadWithExtraDirectly(ByteBuff buff, FSDataInputStream dis, long position, int necessaryLen, int extraLen, boolean readAllBytes) throws IOException {
        int directBytesRead = 0;
        int heapBytesRead = 0;
        int remain = necessaryLen + extraLen;
        int bytesRead = 0;
        int idx = 0;
        ByteBuffer[] buffers = buff.nioByteBuffers();
        ByteBuffer cur = buffers[idx];
        int lengthMustRead = readAllBytes ? remain : necessaryLen;
        try {
            while (bytesRead < lengthMustRead) {
                int ret;
                while (!cur.hasRemaining()) {
                    if (++idx >= buffers.length) {
                        throw new IOException("Not enough ByteBuffers to read the reminding " + remain + "bytes");
                    }
                    cur = buffers[idx];
                }
                cur.limit(cur.position() + Math.min(remain, cur.remaining()));
                try {
                    ret = (Integer)byteBufferPositionedReadMethod.invoke((Object)dis, position + (long)bytesRead, cur);
                }
                catch (IllegalAccessException e) {
                    throw new IOException("Unable to invoke ByteBuffer positioned read when trying to read " + bytesRead + " bytes from position " + position, e);
                }
                catch (InvocationTargetException e) {
                    throw new IOException("Encountered an exception when invoking ByteBuffer positioned read when trying to read " + bytesRead + " bytes from position " + position, e);
                }
                if (ret < 0) {
                    throw new IOException("Premature EOF from inputStream (positional read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + bytesRead);
                }
                bytesRead += ret;
                remain -= ret;
                if (cur.isDirect()) {
                    directBytesRead += bytesRead;
                    continue;
                }
                heapBytesRead += bytesRead;
            }
        }
        finally {
            Span span = Span.current();
            AttributesBuilder attributesBuilder = BlockIOUtils.builderFromContext(Context.current());
            BlockIOUtils.annotateBytesRead(attributesBuilder, directBytesRead, heapBytesRead);
            span.addEvent("BlockIOUtils.preadWithExtra", attributesBuilder.build());
        }
        return extraLen > 0 && bytesRead == necessaryLen + extraLen;
    }

    private static int copyToByteBuff(byte[] buf, int offset, int len, ByteBuff out) throws IOException {
        if (offset < 0 || len < 0 || offset + len > buf.length) {
            throw new IOException("Invalid offset=" + offset + " and len=" + len + ", cap=" + buf.length);
        }
        ByteBuffer[] buffers = out.nioByteBuffers();
        int idx = 0;
        int remain = len;
        ByteBuffer cur = buffers[idx];
        while (remain > 0) {
            while (!cur.hasRemaining()) {
                if (++idx >= buffers.length) {
                    throw new IOException("Not enough ByteBuffers to read the reminding " + remain + "bytes");
                }
                cur = buffers[idx];
            }
            int copyLen = Math.min(cur.remaining(), remain);
            cur.put(buf, offset, copyLen);
            remain -= copyLen;
            offset += copyLen;
        }
        return len;
    }

    private static AttributesBuilder builderFromContext(Context context) {
        AttributesBuilder attributesBuilder = Attributes.builder();
        Optional.ofNullable(context).map(val -> (Consumer)val.get(HFileContextAttributesBuilderConsumer.CONTEXT_KEY)).ifPresent(c -> c.accept(attributesBuilder));
        return attributesBuilder;
    }

    private static void annotateHeapBytesRead(AttributesBuilder attributesBuilder, int heapBytesRead) {
        BlockIOUtils.annotateBytesRead(attributesBuilder, 0L, heapBytesRead);
    }

    private static void annotateBytesRead(AttributesBuilder attributesBuilder, long directBytesRead, long heapBytesRead) {
        if (directBytesRead > 0L) {
            attributesBuilder.put(HBaseSemanticAttributes.DIRECT_BYTES_READ_KEY, (Object)directBytesRead);
        }
        if (heapBytesRead > 0L) {
            attributesBuilder.put(HBaseSemanticAttributes.HEAP_BYTES_READ_KEY, (Object)heapBytesRead);
        }
    }

    static {
        BlockIOUtils.initByteBufferPositionReadableMethod();
    }
}

