/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.lake.committer;

import org.apache.fluss.utils.types.Tuple2;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * The lake already committed snapshot, containing the lake snapshot id and the bucket end offsets
 * in this snapshot.
 */
public class CommittedLakeSnapshot {

    private final long lakeSnapshotId;
    // <partition_id, bucket> -> log offset, partition_id will be null if it's not a
    // partition bucket
    private final Map<Tuple2<Long, Integer>, Long> logEndOffsets = new HashMap<>();

    // partition id -> partition name, will be empty if is not a partitioned table
    // the partition name is a qualified name in the format: key1=value1/key2=value2
    private final Map<Long, String> qualifiedPartitionNameById = new HashMap<>();

    public CommittedLakeSnapshot(long lakeSnapshotId) {
        this.lakeSnapshotId = lakeSnapshotId;
    }

    public long getLakeSnapshotId() {
        return lakeSnapshotId;
    }

    public void addBucket(int bucketId, long offset) {
        logEndOffsets.put(Tuple2.of(null, bucketId), offset);
    }

    public void addPartitionBucket(
            Long partitionId, String partitionQualifiedName, int bucketId, long offset) {
        logEndOffsets.put(Tuple2.of(partitionId, bucketId), offset);
        qualifiedPartitionNameById.put(partitionId, partitionQualifiedName);
    }

    public Map<Tuple2<Long, Integer>, Long> getLogEndOffsets() {
        return logEndOffsets;
    }

    public Map<Long, String> getQualifiedPartitionNameById() {
        return qualifiedPartitionNameById;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CommittedLakeSnapshot that = (CommittedLakeSnapshot) o;
        return lakeSnapshotId == that.lakeSnapshotId
                && Objects.equals(logEndOffsets, that.logEndOffsets)
                && Objects.equals(qualifiedPartitionNameById, that.qualifiedPartitionNameById);
    }

    @Override
    public int hashCode() {
        return Objects.hash(lakeSnapshotId, logEndOffsets, qualifiedPartitionNameById);
    }

    @Override
    public String toString() {
        return "CommittedLakeSnapshot{"
                + "lakeSnapshotId="
                + lakeSnapshotId
                + ", logEndOffsets="
                + logEndOffsets
                + ", partitionNameById="
                + qualifiedPartitionNameById
                + '}';
    }
}
