/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.mcp.tools;

import org.apache.seata.common.util.StringUtils;
import org.apache.seata.mcp.service.ModifyConfirmService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springaicommunity.mcp.annotation.McpTool;
import org.springaicommunity.mcp.annotation.McpToolParam;
import org.springframework.stereotype.Service;

import java.util.Map;

@Service
public class ModifyConfirmTools {

    private static final Logger LOGGER = LoggerFactory.getLogger(ModifyConfirmTools.class);

    private final ModifyConfirmService modifyConfirmService;

    public ModifyConfirmTools(ModifyConfirmService modifyConfirmService) {
        this.modifyConfirmService = modifyConfirmService;
    }

    @McpTool(
            description = "Before modifying (update or delete) a transaction or lock, the user MUST manually confirm."
                    + "You are NOT allowed to fabricate or auto-confirm on behalf of the user.")
    public Map<String, String> confirmAndGetKey(
            @McpToolParam(
                            description =
                                    "The confirmation string provided by the USER (not generated by the LLM).The content must repeat the modification action clearly.")
                    String userInputStr) {
        if (StringUtils.isBlank(userInputStr)) {
            throw new IllegalArgumentException("User confirmation string is required.");
        }
        if (!userInputStr.contains("确认") && !userInputStr.contains("confirm")) {
            throw new IllegalArgumentException(
                    "Confirmation string must explicitly contain '确认' or 'confirm' and repeat the modification content. This must come from the user.");
        }
        Map<String, String> keyMap = modifyConfirmService.confirmAndGetKey();
        LOGGER.info("the user obtains a modify key:{}", keyMap.get("modify_key"));
        return keyMap;
    }
}
