/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.graph.compose;

import org.apache.jena.graph.*;
import org.apache.jena.graph.impl.SimpleEventManager;
import org.apache.jena.util.iterator.ExtendedIterator;

/**
 * Base class for the two-operand composition operations; has two graphs L and R
 */

public abstract class Dyadic extends CompositionBase {
    protected Graph L;
    protected Graph R;

    /**
     * When the graph is constructed, copy the prefix mappings of both components
     * into this prefix mapping. The prefix mapping doesn't change afterwards with
     * the components, which might be regarded as a bug.
     */
    public Dyadic(Graph L, Graph R) {
        this.L = L;
        this.R = R;
        getPrefixMapping().setNsPrefixes(L.getPrefixMapping()).setNsPrefixes(R.getPrefixMapping());
    }

    /**
     * override graphBaseFind to return an iterator that will report when a deletion
     * occurs.
     */
    @Override
    protected final ExtendedIterator<Triple> graphBaseFind(Triple m) {
        return SimpleEventManager.notifyingRemove(this, this._graphBaseFind(m));
    }

    /**
     * The method that the overridden graphBaseFind( Triple t ) calls to actually do
     * the work of finding.
     */
    protected abstract ExtendedIterator<Triple> _graphBaseFind(Triple t);

    @Override
    public void close() {
        L.close();
        R.close();
        this.closed = true;
    }

    public Union union(Graph X) {
        return new Union(this, X);
    }

    /**
     * Answer the left (first) operand of this Dyadic.
     */
    public Graph getL() {
        return L;
    }

    /**
     * Answer the right (second) operand of this Dyadic.
     */
    public Graph getR() {
        return R;
    }

}
