/***************************************************************************
 * arch/arm64/src/common/arm64_gicv3.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ***************************************************************************/

/***************************************************************************
 * Included Files
 ***************************************************************************/

#include <nuttx/config.h>
#include <debug.h>
#include <assert.h>

#include <nuttx/arch.h>
#include <nuttx/compiler.h>
#include <arch/barriers.h>
#include <arch/irq.h>
#include <arch/chip/chip.h>
#include <sched/sched.h>

#include "arm64_arch.h"
#include "arm64_internal.h"
#include "arm64_gic.h"
#include "arm64_fatal.h"

/***************************************************************************
 * Pre-processor Definitions
 ***************************************************************************/

#define GICR_TYPER_NR_PPIS(r)                   \
  ({                                            \
    unsigned int __ppinum = ((r) >> 27) & 0x1f; \
    unsigned int __nr_ppis = 16;                \
    if (__ppinum == 1 || __ppinum == 2)         \
        {  __nr_ppis +=  __ppinum * 32;  }      \
    __nr_ppis;                                  \
  })

/* selects redistributor SGI_base for current core for PPI and SGI
 * selects distributor base for SPI
 * The macro translates to distributor base for GICv2 and GICv1
 */

#define GET_DIST_BASE(intid)  ((intid < GIC_SPI_INT_BASE) ?          \
                               (gic_get_rdist() + GICR_SGI_BASE_OFF) \
                               : GIC_DIST_BASE)

#define IGROUPR_VAL  0xFFFFFFFFU

#ifdef CONFIG_ARM64_DECODEFIQ

/* Config SGI8 ~ SGI15 as group0, to signal fiq */

#  define IGROUPR_SGI_VAL 0xFFFF00FFU
#else
#  define IGROUPR_SGI_VAL 0xFFFFFFFFU
#endif

#define PENDING_GRP1NS_INTID 1021
#define SPURIOUS_INT         1023

/***************************************************************************
 * Private Data
 ***************************************************************************/

/* Redistributor base addresses for each core */

static unsigned long g_gic_rdists[CONFIG_SMP_NCPUS];
static volatile spinlock_t g_gic_lock = SP_UNLOCKED;

/***************************************************************************
 * Private Functions
 ***************************************************************************/

unused_code static inline void sys_set_bit(unsigned long addr,
                                           unsigned int bit)
{
  uint32_t temp;

  temp = getreg32(addr);
  temp = temp | (BIT(bit));
  putreg32(temp, addr);
}

static inline void sys_clear_bit(unsigned long addr, unsigned int bit)
{
  uint32_t temp;

  temp = getreg32(addr);
  temp = temp & ~(BIT(bit));
  putreg32(temp, addr);
}

unused_code static inline int sys_test_bit(unsigned long addr,
                                           unsigned int bit)
{
  uint32_t temp;

  temp = getreg32(addr);
  return (temp & BIT(bit));
}

static inline unsigned long gic_get_rdist(void)
{
  return g_gic_rdists[this_cpu()];
}

/* Wait for register write pending
 * TODO: add timed wait
 */

static int gic_wait_rwp(uint32_t intid)
{
  uint32_t      rwp_mask;
  unsigned long base;

  if (intid < GIC_SPI_INT_BASE)
    {
      base        = (gic_get_rdist() + GICR_CTLR);
      rwp_mask    = BIT(GICR_CTLR_RWP);
    }
  else
    {
      base        = GICD_CTLR;
      rwp_mask    = BIT(GICD_CTLR_RWP);
    }

  while (getreg32(base) & rwp_mask)
    {
    }

  return 0;
}

static inline void arm64_gic_write_irouter(uint64_t val, unsigned int intid)
{
  unsigned long addr = IROUTER(GET_DIST_BASE(intid), intid);

  putreg64(val, addr);
}

void arm64_gic_irq_set_priority(unsigned int intid, unsigned int prio,
                                uint32_t flags)
{
  uint32_t      mask  = BIT(intid & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t      idx   = intid / GIC_NUM_INTR_PER_REG;
  uint32_t      shift;
  uint32_t      val;
  unsigned long base = GET_DIST_BASE(intid);
  irqstate_t    irq_flags;

  /* Disable the interrupt */

  putreg32(mask, ICENABLER(base, idx));
  gic_wait_rwp(intid);

  /* PRIORITYR registers provide byte access */

  putreg8(prio & GIC_PRI_MASK, IPRIORITYR(base, intid));

  /* Interrupt type config */

  if (!GIC_IS_SGI(intid))
    {
      idx     = intid / GIC_NUM_CFG_PER_REG;
      shift   = (intid & (GIC_NUM_CFG_PER_REG - 1)) * 2;

      /* GICD_ICFGR requires full 32-bit RMW operations.
       * Each interrupt uses 2 bits; thus updates must be synchronized
       * to avoid losing configuration in concurrent environments.
       */

      irq_flags = spin_lock_irqsave(&g_gic_lock);

      val = getreg32(ICFGR(base, idx));
      val &= ~(GICD_ICFGR_MASK << shift);
      if (flags & IRQ_TYPE_EDGE)
        {
          val |= (GICD_ICFGR_TYPE << shift);
        }

      putreg32(val, ICFGR(base, idx));
      spin_unlock_irqrestore(&g_gic_lock, irq_flags);
    }
}

/***************************************************************************
 * Name: arm64_gic_irq_trigger
 *
 * Description:
 *   Set the trigger type for the specified IRQ source and the current CPU.
 *
 *   Since this API is not supported on all architectures, it should be
 *   avoided in common implementations where possible.
 *
 * Input Parameters:
 *   irq   - The interrupt request to modify.
 *   flags - irq type, IRQ_TYPE_EDGE or IRQ_TYPE_LEVEL
 *           Default is IRQ_TYPE_LEVEL
 *
 * Returned Value:
 *   Zero (OK) on success; a negated errno value is returned on any failure.
 *
 ***************************************************************************/

int arm64_gic_irq_trigger(unsigned int intid, uint32_t flags)
{
  uint32_t      idx  = intid / GIC_NUM_INTR_PER_REG;
  uint32_t      shift;
  uint32_t      val;
  unsigned long base = GET_DIST_BASE(intid);
  irqstate_t    irq_flags;

  if (!GIC_IS_SGI(intid))
    {
      idx   = intid / GIC_NUM_CFG_PER_REG;
      shift = (intid & (GIC_NUM_CFG_PER_REG - 1)) * 2;

      /* GICD_ICFGR requires full 32-bit RMW operations.
       * Each interrupt uses 2 bits; thus updates must be synchronized
       * to avoid losing configuration in concurrent environments.
       */

      irq_flags = spin_lock_irqsave(&g_gic_lock);

      val = getreg32(ICFGR(base, idx));
      val &= ~(GICD_ICFGR_MASK << shift);
      if (flags & IRQ_TYPE_EDGE)
        {
          val |= (GICD_ICFGR_TYPE << shift);
        }

      putreg32(val, ICFGR(base, idx));
      spin_unlock_irqrestore(&g_gic_lock, irq_flags);
      return OK;
    }

  return -EINVAL;
}

void arm64_gic_irq_enable(unsigned int intid)
{
  uint32_t mask = BIT(intid & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t idx  = intid / GIC_NUM_INTR_PER_REG;

  /* Affinity routing is enabled for Non-secure state (GICD_CTLR.ARE_NS
   * is set to '1' when GIC distributor is initialized) ,so need to set
   * SPI's affinity, now set it to be the PE on which it is enabled.
   */

#ifndef CONFIG_ARM64_GICV3_SPI_ROUTING_CPU0
  if (GIC_IS_SPI(intid))
    {
      arm64_gic_write_irouter(up_cpu_index(), intid);
    }
#endif

  putreg32(mask, ISENABLER(GET_DIST_BASE(intid), idx));
}

void arm64_gic_irq_disable(unsigned int intid)
{
  uint32_t mask = BIT(intid & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t idx  = intid / GIC_NUM_INTR_PER_REG;

  putreg32(mask, ICENABLER(GET_DIST_BASE(intid), idx));

  /* poll to ensure write is complete */

  gic_wait_rwp(intid);
}

bool arm64_gic_irq_is_enabled(unsigned int intid)
{
  uint32_t mask = BIT(intid & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t idx  = intid / GIC_NUM_INTR_PER_REG;
  uint32_t val;

  val = getreg32(ISENABLER(GET_DIST_BASE(intid), idx));

  return (val & mask) != 0;
}

unsigned int arm64_gic_get_active_irq(void)
{
  int intid;

  /* (Pending -> Active / AP) or (AP -> AP) */

  intid = read_sysreg(ICC_IAR1_EL1);

  /* The ARM GICv3 specification states in 4.1.1 Physical CPU Interface:
   * The effects of reading ICC_IAR0_EL1 and ICC_IAR1_EL1
   * on the state of a returned INTID are not guaranteed
   * to be visible until after the execution of a DSB.
   */

  UP_DSB();
  return intid;
}

#ifdef CONFIG_ARM64_DECODEFIQ
unsigned int arm64_gic_get_active_fiq(void)
{
  int intid;

  /* (Pending -> Active / AP) or (AP -> AP) */

  intid = read_sysreg(ICC_IAR0_EL1);

  /* The ARM GICv3 specification states in 4.1.1 Physical CPU Interface:
   * The effects of reading ICC_IAR0_EL1 and ICC_IAR1_EL1
   * on the state of a returned INTID are not guaranteed
   * to be visible until after the execution of a DSB.
   */

  UP_DSB();
  return intid;
}
#endif

void aarm64_gic_eoi_irq(unsigned int intid)
{
  /* Interrupt request deassertion from peripheral to GIC happens
   * by clearing interrupt condition by a write to the peripheral
   * register. It is desired that the write transfer is complete
   * before the core tries to change GIC state from 'AP/Active' to
   * a new state on seeing 'EOI write'.
   * Since ICC interface writes are not ordered against Device
   * memory writes, a barrier is required to ensure the ordering.
   * The dsb will also ensure *completion* of previous writes with
   * DEVICE nGnRnE attribute.
   */

  UP_DSB();

  /* (AP -> Pending) Or (Active -> Inactive) or (AP to AP) nested case */

  write_sysreg(intid, ICC_EOIR1_EL1);

  UP_ISB();
}

#ifdef CONFIG_ARM64_DECODEFIQ
void arm64_gic_eoi_fiq(unsigned int intid)
{
  /* Interrupt request deassertion from peripheral to GIC happens
   * by clearing interrupt condition by a write to the peripheral
   * register. It is desired that the write transfer is complete
   * before the core tries to change GIC state from 'AP/Active' to
   * a new state on seeing 'EOI write'.
   * Since ICC interface writes are not ordered against Device
   * memory writes, a barrier is required to ensure the ordering.
   * The dsb will also ensure *completion* of previous writes with
   * DEVICE nGnRnE attribute.
   */

  UP_DSB();

  /* (AP -> Pending) Or (Active -> Inactive) or (AP to AP) nested case */

  write_sysreg(intid, ICC_EOIR0_EL1);
  UP_ISB();
}
#endif

static int arm64_gic_send_sgi(unsigned int sgi_id, uint64_t target_aff,
                              uint16_t target_list)
{
  uint32_t aff3;
  uint32_t aff2;
  uint32_t aff1;
  uint64_t sgi_val;
  uint32_t regval;

  ASSERT(GIC_IS_SGI(sgi_id));

  /* Extract affinity fields from target */

  aff1 = MPIDR_AFFLVL(target_aff, 1);
  aff2 = MPIDR_AFFLVL(target_aff, 2);
  aff3 = MPIDR_AFFLVL(target_aff, 3);

  sgi_val = GICV3_SGIR_VALUE(aff3, aff2, aff1, sgi_id, SGIR_IRM_TO_AFF,
                             target_list);

  UP_DSB();

  /* Read the IGROUPR0 value we set in `gicv3_cpuif_init` */

  regval = IGROUPR_SGI_VAL;

  if (regval & BIT(sgi_id))
    {
      write_sysreg(sgi_val, ICC_SGI1R);     /* Group 1 */
    }
  else
    {
      write_sysreg(sgi_val, ICC_SGI0R_EL1); /* Group 0 */
    }

  UP_ISB();

  return 0;
}

void arm64_gic_raise_sgi(unsigned int sgi_id, uint16_t target_list)
{
  uint64_t pre_cluster_id = UINT64_MAX;
  uint64_t curr_cluster_id;
  uint64_t curr_mpidr;
  uint16_t tlist = 0;
  uint16_t cpu = 0;
  uint16_t i;

  while ((i = ffs(target_list)))
    {
      target_list >>= i;

      curr_mpidr = arm64_get_mpid(cpu + (i - 1));
      curr_cluster_id = MPID_TO_CLUSTER_ID(curr_mpidr);

      if (pre_cluster_id != UINT64_MAX &&
          pre_cluster_id != curr_cluster_id)
        {
          arm64_gic_send_sgi(sgi_id, pre_cluster_id, tlist);
        }

      tlist |= 1 << (curr_mpidr & MPIDR_AFFLVL_MASK);

      cpu += i;
      pre_cluster_id = curr_cluster_id;
    }

  arm64_gic_send_sgi(sgi_id, pre_cluster_id, tlist);
}

/* Wake up GIC redistributor.
 * clear ProcessorSleep and wait till ChildAsleep is cleared.
 * ProcessSleep to be cleared only when ChildAsleep is set
 * Check if redistributor is not powered already.
 */

static void gicv3_rdist_enable(unsigned long rdist)
{
  if (!(getreg32(rdist + GICR_WAKER) & BIT(GICR_WAKER_CA)))
    {
      return;
    }

  /* Power up sequence of the Redistributors for GIC600/GIC700
   * please check GICR_PWRR define at trm of GIC600/GIC700
   */

  putreg32(0x2, rdist + GICR_PWRR);

  sys_clear_bit(rdist + GICR_WAKER, GICR_WAKER_PS);

  while (getreg32(rdist + GICR_WAKER) & BIT(GICR_WAKER_CA))
    {
    }
}

/* Initialize the cpu interface. This should be called by each core. */

static void gicv3_cpuif_init(void)
{
  uint64_t      icc_sre;
  uint64_t      icc_ctrl;
  uint32_t      intid;
  unsigned long base = gic_get_rdist() + GICR_SGI_BASE_OFF;
  bool eoi_mode;

  /* Disable all SGI and PPI interrupts */

  putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICENABLER(base, 0));

  /* Any sgi/ppi intid ie. 0-31 will select GICR_CTRL */

  gic_wait_rwp(0);

  /* Clear pending and active SGI and PPI interrupts at GIC */

  putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICPENDR(base, 0));
  putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICACTIVER(base, 0));

  /* Configure all SGIs/PPIs as G1S or G1NS depending on NuttX is run in
   * EL1S or EL1NS respectively. All interrupts will be delivered as irq.
   */

  putreg32(IGROUPR_SGI_VAL, IGROUPR(base, 0));
  putreg32(0, IGROUPMODR(base, 0));

  /* Clear any active IRQs in the CPU interface */

  icc_ctrl = read_sysreg(ICC_CTLR_EL1);
  UP_DSB();
  eoi_mode = (icc_ctrl & ICC_CTLR_EOIMODE_BIT) != 0;

  /* If the SW has crashed / warm-rebooted right after reading the
   * ICC_IAR1_EL1, but before acking it in EOIR1, the irq is still
   * active, but not any longer readable from the IAR1.
   *
   * Just ack all IRQs to clear such state. Invalid writes are ignored by
   * the architecture.
   */

  for (intid = 0; intid < NR_IRQS; intid++)
    {
#ifdef CONFIG_ARM64_DECODEFIQ
      write_sysreg(intid, ICC_EOIR0_EL1);
#endif
      write_sysreg(intid, ICC_EOIR1_EL1);

      if (eoi_mode)
        {
          write_sysreg(intid, ICC_DIR_EL1);
        }
    }

  /* Now, if the SW has crashed / warm rebooted when interrupts are
   * activated, but before reading the ICC_IAR1_EL1, disable the
   * interrupts normally
   */

  while ((intid = arm64_gic_get_active_irq()) < NR_IRQS)
    {
#ifdef CONFIG_ARM64_DECODEFIQ
      write_sysreg(intid, ICC_EOIR0_EL1);
#endif
      write_sysreg(intid, ICC_EOIR1_EL1);
      if (eoi_mode)
        {
          write_sysreg(intid, ICC_DIR_EL1);
        }
    }

  /* Configure default priorities for SGI 0:15 and PPI 0:15. */

  for (intid = 0; intid < GIC_SPI_INT_BASE;
       intid += GIC_NUM_PRI_PER_REG)
    {
      putreg32(GIC_INT_DEF_PRI_X4, IPRIORITYR(base, intid));
    }

  /* Configure PPIs as level triggered */

  putreg32(0, ICFGR(base, 1));

  /* Check if system interface can be enabled.
   * 'icc_sre_el3' needs to be configured at 'EL3'
   * to allow access to 'icc_sre_el1' at 'EL1'
   */

  icc_sre = read_sysreg(ICC_SRE_EL1);

  if (!(icc_sre & ICC_SRE_ELX_SRE_BIT))
    {
      icc_sre =
        (icc_sre | ICC_SRE_ELX_SRE_BIT | ICC_SRE_ELX_DIB_BIT |
         ICC_SRE_ELX_DFB_BIT);
      write_sysreg(icc_sre, ICC_SRE_EL1);

      UP_ISB();

      icc_sre = read_sysreg(ICC_SRE_EL1);

      ASSERT(icc_sre & ICC_SRE_ELX_SRE_BIT);
    }

  write_sysreg(GIC_IDLE_PRIO, ICC_PMR_EL1);

  /* Allow group1 interrupts */

  write_sysreg(1, ICC_IGRPEN1_EL1);

#ifdef CONFIG_ARM64_DECODEFIQ
  write_sysreg(1, ICC_IGRPEN0_EL1);
#endif

  UP_ISB();
}

static void gicv3_dist_init(void)
{
  unsigned int  num_ints;
  unsigned int  intid;
  unsigned int  idx;
  unsigned int  regval;
  unsigned long base = GIC_DIST_BASE;

  num_ints  = getreg32(GICD_TYPER);
  num_ints  &= GICD_TYPER_ITLINESNUM_MASK;
  num_ints  = (num_ints + 1) << 5;

  /* Disable the distributor */

  putreg32(0, GICD_CTLR);
  gic_wait_rwp(GIC_SPI_INT_BASE);

#ifdef CONFIG_ARCH_SINGLE_SECURITY_STATE

  /* Before configuration, we need to check whether
   * the GIC single security state mode is supported.
   * Make sure GICD_CTRL_NS is 1.
   */

  sys_set_bit(GICD_CTLR, GICD_CTRL_DS);
  if (!sys_test_bit(GICD_CTLR, GICD_CTRL_DS))
    {
      sinfo("Current GIC does not support single security state\n");
      PANIC();
    }
#endif

  /* Default configuration of all SPIs */

  for (intid = GIC_SPI_INT_BASE; intid < num_ints;
       intid += GIC_NUM_INTR_PER_REG)
    {
      idx = intid / GIC_NUM_INTR_PER_REG;

      /* Disable interrupt */

      putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICENABLER(base, idx));

      /* Wait for rwp on GICD */

      gic_wait_rwp(intid);

      /* Clear pending and active SPIs */

      putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICPENDR(base, idx));
      putreg32(BIT64_MASK(GIC_NUM_INTR_PER_REG), ICACTIVER(base, idx));

      /* Configure groups to default values */

      putreg32(IGROUPR_VAL, IGROUPR(base, idx));
      putreg32(0, IGROUPMODR(base, idx));
    }

  /* Configure default priorities for all SPIs. */

  for (intid = GIC_SPI_INT_BASE; intid < num_ints;
       intid += GIC_NUM_PRI_PER_REG)
    {
      putreg32(GIC_INT_DEF_PRI_X4, IPRIORITYR(base, intid));
    }

  /* Configure all SPIs as active low, level triggered by default */

  for (intid = GIC_SPI_INT_BASE; intid < num_ints;
       intid += GIC_NUM_CFG_PER_REG)
    {
      idx = intid / GIC_NUM_CFG_PER_REG;
#ifdef CONFIG_ARM64_GICV3_SPI_EDGE
      /* Configure all SPIs as edge-triggered by default */

      putreg32(0xaaaaaaaa, ICFGR(base, idx));
#else
      /* Configure all SPIs as level-sensitive by default */

      putreg32(0, ICFGR(base, idx));
#endif
    }

  /* Configure SPI interrupt affinity routing to CPU0 */

#ifdef CONFIG_ARM64_GICV3_SPI_ROUTING_CPU0
  uint64_t mpid = arm64_get_mpid(0);

  for (intid = GIC_SPI_INT_BASE; intid < num_ints; intid++)
    {
      putreg64(mpid, IROUTER(base, intid));
    }
#endif

  /* TODO: Some arrch64 Cortex-A core maybe without security state
   * it has different GIC configure with standard arrch64 A or R core
   */

#ifdef CONFIG_ARCH_SINGLE_SECURITY_STATE
  /* For GIC single security state(ARMv8-R), the config means
   * the GIC is under single security state which has
   * only two groups:
   *  group 0 and group 1.
   * Then set GICD_CTLR_ARE and GICD_CTLR_ENABLE_G1 to enable Group 1
   * interrupt.
   * Since the GICD_CTLR_ARE and GICD_CTRL_ARE_S share BIT(4), and
   * similarly the GICD_CTLR_ENABLE_G1 and GICD_CTLR_ENABLE_G1NS share
   * BIT(1), we can reuse them.
   */

  regval = BIT(GICD_CTRL_ARE_S) | BIT(GICD_CTLR_ENABLE_G1NS);

#ifdef CONFIG_ARM64_DECODEFIQ
  regval |= BIT(GICD_CTLR_ENABLE_G0);
#endif

  putreg32(regval, GICD_CTLR);

#else
  /* Enable distributor with ARE */

  regval = BIT(GICD_CTRL_ARE_NS) | BIT(GICD_CTLR_ENABLE_G1NS);

#ifdef CONFIG_ARM64_DECODEFIQ
  regval |= BIT(GICD_CTLR_ENABLE_G0);
#endif

  putreg32(regval, GICD_CTLR);
#endif

#ifdef CONFIG_SMP
  /* Attach SGI interrupt handlers. This attaches the handler to all CPUs. */

  DEBUGVERIFY(irq_attach(GIC_SMP_SCHED, arm64_smp_sched_handler, NULL));
  DEBUGVERIFY(irq_attach(GIC_SMP_CALL, nxsched_smp_call_handler, NULL));
#endif
}

void up_enable_irq(int irq)
{
  arm64_gic_irq_enable(irq);
}

void up_disable_irq(int irq)
{
  arm64_gic_irq_disable(irq);
}

/***************************************************************************
 * Name: up_set_secure_irq
 *
 * Description:
 *   Secure an IRQ
 *
 ***************************************************************************/

#if defined(CONFIG_ARCH_TRUSTZONE_SECURE) || defined(CONFIG_ARCH_HIPRI_INTERRUPT)
void up_secure_irq(int irq, bool secure)
{
  uint32_t mask      = BIT(irq & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t idx       = irq / GIC_NUM_INTR_PER_REG;
  unsigned long base = GET_DIST_BASE(irq);
  unsigned int val   = getreg32(IGROUPR(base, idx));

  if (secure)
    {
      val &= (~mask);  /* group 0 fiq */
    }
  else
    {
      val |= mask;     /* group 1 irq */
    }

  putreg32(val, IGROUPR(base, idx));
}
#endif

/***************************************************************************
 * Name: up_prioritize_irq
 *
 * Description:
 *   Set the priority of an IRQ.
 *
 *   Since this API is not supported on all architectures, it should be
 *   avoided in common implementations where possible.
 *
 ***************************************************************************/

int up_prioritize_irq(int irq, int priority)
{
  unsigned long base = GET_DIST_BASE(irq);

  DEBUGASSERT(irq >= 0 && irq < NR_IRQS &&
              priority >= 0 && priority <= 255);

  /* Ignore invalid interrupt IDs */

  if (irq >= 0 && irq < NR_IRQS)
    {
      /* PRIORITYR registers provide byte access */

      putreg8(priority & GIC_PRI_MASK, IPRIORITYR(base, irq));
      return OK;
    }

  return -EINVAL;
}

/***************************************************************************
 * Name: up_affinity_irq
 *
 * Description:
 *   Set an IRQ affinity by software.
 *
 ***************************************************************************/

void up_affinity_irq(int irq, cpu_set_t cpuset)
{
  if (GIC_IS_SPI(irq))
    {
      /* Only support interrupt routing mode 0,
       * so routing to the first cpu in cpuset.
       */

      uint64_t mpid = arm64_get_mpid(ffs(cpuset) - 1);
      arm64_gic_write_irouter(mpid, irq);
    }
}

/***************************************************************************
 * Name: up_trigger_irq
 *
 * Description:
 *   Perform a Software Generated Interrupt (SGI).  If CONFIG_SMP is
 *   selected, then the SGI is sent to all CPUs specified in the CPU set.
 *   That set may include the current CPU.
 *
 *   If CONFIG_SMP is not selected, the cpuset is ignored and SGI is sent
 *   only to the current CPU.
 *
 * Input Parameters
 *   irq    - The SGI interrupt ID (0-15)
 *   cpuset - The set of CPUs to receive the SGI
 *
 ***************************************************************************/

void up_trigger_irq(int irq, cpu_set_t cpuset)
{
  uint32_t  mask  = BIT(irq & (GIC_NUM_INTR_PER_REG - 1));
  uint32_t  idx   = irq / GIC_NUM_INTR_PER_REG;

  if (GIC_IS_SGI(irq))
    {
      arm64_gic_raise_sgi(irq, cpuset);
    }
  else if (irq >= 0 && irq < NR_IRQS)
    {
      /* Write '1' to the corresponding bit in the distributor Interrupt
       * Set-Pending (ISPENDR)
       * GICD_ISPENDRn: Interrupt Set-Pending Registers
       */

      putreg32(mask, ISPENDR(GET_DIST_BASE(irq), idx));
    }
}

/***************************************************************************
 * Name: arm64_decodeirq
 *
 * Description:
 *   This function is called from the IRQ vector handler in arm64_vectors.S.
 *   At this point, the interrupt has been taken and the registers have
 *   been saved on the stack.  This function simply needs to determine the
 *   the irq number of the interrupt and then to call arm_doirq to dispatch
 *   the interrupt.
 *
 *  Input Parameters:
 *   regs - A pointer to the register save area on the stack.
 ***************************************************************************/

uint64_t * arm64_decodeirq(uint64_t * regs)
{
  int irq;

  /* Read the interrupt acknowledge register and get the interrupt ID */

  irq = arm64_gic_get_active_irq();

  /* Ignore spurions IRQs.  ICCIAR will report 1023 if there is no pending
   * interrupt.
   */

  DEBUGASSERT(irq < NR_IRQS || irq == SPURIOUS_INT);
  if (irq < NR_IRQS)
    {
      /* Dispatch the interrupt */

      regs = arm64_doirq(irq, regs);
    }

  /* Write to the end-of-interrupt register */

  aarm64_gic_eoi_irq(irq);

  return regs;
}

#ifdef CONFIG_ARM64_DECODEFIQ
uint64_t * arm64_decodefiq(uint64_t * regs)
{
  int irq;

  /* Read the interrupt acknowledge register and get the interrupt ID */

  irq = arm64_gic_get_active_fiq();

#if CONFIG_ARCH_ARM64_EXCEPTION_LEVEL == 3
  /* FIQ is group0 interrupt */

  if (irq == PENDING_GRP1NS_INTID)
    {
      /* irq 1021 indicates that the irq being acked is expected at EL1/EL2.
       * However, EL3 has no interrupts, only FIQs, see:
       * 'Arm® Generic Interrupt Controller, Architecture Specification GIC
       *  architecture version 3 and version 4' Arm IHI 0069G (ID011821)
       * 'Table 4-3 Interrupt signals for two Security states when EL3 is
       *  using AArch64 state'
       *
       * Thus we know there's an interrupt so let's handle it from group1.
       */

      regs = arm64_decodeirq(regs);
      arm64_gic_eoi_fiq(irq);

      return regs;
    }
#endif

  /* Ignore spurions IRQs.  ICCIAR will report 1023 if there is no pending
   * interrupt.
   */

  DEBUGASSERT(irq < NR_IRQS || irq == SPURIOUS_INT);
  if (irq < NR_IRQS)
    {
      /* Dispatch the interrupt */

      regs = arm64_doirq(irq, regs);
    }

  /* Write to the end-of-interrupt register */

  arm64_gic_eoi_fiq(irq);

  return regs;
}
#endif

static int gic_validate_dist_version(void)
{
  uint32_t typer;
  bool     has_rss;
  uint32_t reg = getreg32(GICD_PIDR2) & GICD_PIDR2_ARCH_MASK;
  int      spis;
  int      espis;

  if (reg == GICD_PIDR2_ARCH_GICV3)
    {
      sinfo("GICv3 version detect\n");
    }
  else if (reg == GICD_PIDR2_ARCH_GICV4)
    {
      sinfo("GICv4 version detect\n");
    }
  else
    {
      serr("No GIC version detect\n");
      return -ENODEV;
    }

  /* Find out how many interrupts are supported. */

  typer = getreg32(GICD_TYPER);
  spis  = MIN(GICD_TYPER_SPIS(typer), 1020U) - 32;
  espis = GICD_TYPER_ESPIS(typer);

  sinfo("GICD_TYPER = 0x%x\n", typer);
  sinfo("%d SPIs implemented\n", spis);
  sinfo("%d Extended SPIs implemented\n", espis);

  has_rss = !!(typer & GICD_TYPER_RSS);
  sinfo("Distributor has %sRange Selector support\n", has_rss ? "" : "no ");

  if (typer & GICD_TYPER_MBIS)
    {
      swarn("MBIs is present, But No support\n");
    }

  return 0;
}

static int gic_validate_redist_version(void)
{
  uint64_t      typer;
  unsigned int  ppi_nr;
  bool          has_vlpis       = true;
  bool          has_direct_lpi  = true;
  uint32_t      reg;
  unsigned long redist_base = gic_get_rdist();

  ppi_nr    = (~0U);
  reg       = getreg32(redist_base +
             GICR_PIDR2) & GICR_PIDR2_ARCH_MASK;
  if (reg != GICR_PIDR2_ARCH_GICV3 &&
             reg != GICR_PIDR2_ARCH_GICV4)
    {
      serr("No redistributor present 0x%lx\n", redist_base);
      return -ENODEV;
    }

  typer           = getreg64(redist_base + GICR_TYPER);
  has_vlpis      &= !!(typer & GICR_TYPER_VLPIS);
  has_direct_lpi &= !!(typer & GICR_TYPER_DIRECTLPIS);
  ppi_nr          = MIN(GICR_TYPER_NR_PPIS(typer), ppi_nr);

  if (ppi_nr == (~0U))
    {
      ppi_nr = 0;
    }

  sinfo("GICR_TYPER = 0x%"PRIx64"\n", typer);
  sinfo("%d PPIs implemented\n", ppi_nr);
  sinfo("%sVLPI support, %sdirect LPI support\n", !has_vlpis ? "no " : "",
        !has_direct_lpi ? "no " : "");

  return 0;
}

static void arm64_gic_init(void)
{
  uint8_t   cpu;
  int       err;

  cpu               = this_cpu();
  g_gic_rdists[cpu] = CONFIG_GICR_BASE +
                      up_cpu_index() * CONFIG_GICR_OFFSET;

  err = gic_validate_redist_version();
  if (err)
    {
      swarn("no redistributor detected, giving up ret=%d\n", err);
      return;
    }

  gicv3_rdist_enable(gic_get_rdist());

  gicv3_cpuif_init();

#ifdef CONFIG_SMP
  up_enable_irq(GIC_SMP_SCHED);
  up_enable_irq(GIC_SMP_CALL);
#endif
}

int arm64_gic_initialize(void)
{
  int err;

  err = gic_validate_dist_version();
  if (err)
    {
      swarn("no distributor detected, giving up ret=%d\n", err);
      return err;
    }

  gicv3_dist_init();

  arm64_gic_init();

  return 0;
}

#ifdef CONFIG_SMP
void arm64_gic_secondary_init(void)
{
  arm64_gic_init();
}
#endif

/***************************************************************************
 * Name: up_get_legacy_irq
 *
 * Description:
 *   Reserve vector for legacy
 *
 ***************************************************************************/

int up_get_legacy_irq(uint32_t devfn, uint8_t line, uint8_t pin)
{
  return -ENOTSUP;
}
