/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.util;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.spark.internal.LogKeys;
import org.apache.spark.internal.MDC;
import org.apache.spark.internal.SparkLogger;
import org.apache.spark.internal.SparkLoggerFactory;
import org.apache.spark.network.util.ByteUnit;

public class JavaUtils {
    private static final SparkLogger logger = SparkLoggerFactory.getLogger(JavaUtils.class);
    public static final long DEFAULT_DRIVER_MEM_MB = 1024L;
    private static final Map<String, TimeUnit> timeSuffixes = Map.of("us", TimeUnit.MICROSECONDS, "ms", TimeUnit.MILLISECONDS, "s", TimeUnit.SECONDS, "m", TimeUnit.MINUTES, "min", TimeUnit.MINUTES, "h", TimeUnit.HOURS, "d", TimeUnit.DAYS);
    private static final Map<String, ByteUnit> byteSuffixes = Map.ofEntries(Map.entry("b", ByteUnit.BYTE), Map.entry("k", ByteUnit.KiB), Map.entry("ki", ByteUnit.KiB), Map.entry("kb", ByteUnit.KiB), Map.entry("kib", ByteUnit.KiB), Map.entry("m", ByteUnit.MiB), Map.entry("mi", ByteUnit.MiB), Map.entry("mb", ByteUnit.MiB), Map.entry("mib", ByteUnit.MiB), Map.entry("g", ByteUnit.GiB), Map.entry("gi", ByteUnit.GiB), Map.entry("gb", ByteUnit.GiB), Map.entry("gib", ByteUnit.GiB), Map.entry("t", ByteUnit.TiB), Map.entry("ti", ByteUnit.TiB), Map.entry("tb", ByteUnit.TiB), Map.entry("tib", ByteUnit.TiB), Map.entry("p", ByteUnit.PiB), Map.entry("pi", ByteUnit.PiB), Map.entry("pb", ByteUnit.PiB), Map.entry("pib", ByteUnit.PiB));
    private static final Pattern TIME_STRING_PATTERN = Pattern.compile("(-?[0-9]+)([a-z]+)?");
    private static final Pattern BYTE_STRING_PATTERN = Pattern.compile("([0-9]+)([a-z]+)?");
    private static final Pattern BYTE_STRING_FRACTION_PATTERN = Pattern.compile("([0-9]+\\.[0-9]+)([a-z]+)?");
    public static String osName = System.getProperty("os.name");
    public static String osVersion = System.getProperty("os.version");
    public static String javaVersion = Runtime.version().toString();
    public static String osArch = System.getProperty("os.arch");
    public static boolean isWindows = osName.regionMatches(true, 0, "Windows", 0, 7);
    public static boolean isMac = osName.regionMatches(true, 0, "Mac OS X", 0, 8);
    public static boolean isMacOnAppleSilicon = isMac && osArch.equals("aarch64");
    public static boolean isLinux = osName.regionMatches(true, 0, "Linux", 0, 5);
    public static boolean isUnix = Stream.of("AIX", "HP-UX", "Irix", "Linux", "Mac OS X", "Solaris", "SunOS", "FreeBSD", "OpenBSD", "NetBSD").anyMatch(prefix -> osName.regionMatches(true, 0, (String)prefix, 0, prefix.length()));

    public static void closeQuietly(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        }
        catch (IOException e) {
            logger.error("IOException should not have been thrown.", e);
        }
    }

    public static void deleteQuietly(File file) {
        if (file != null && file.exists()) {
            Path path = file.toPath();
            try (Stream<Path> walk = Files.walk(path, new FileVisitOption[0]);){
                walk.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    public static void forceDeleteOnExit(File file) throws IOException {
        if (file != null && file.exists()) {
            if (!file.isDirectory()) {
                file.deleteOnExit();
            } else {
                Path path = file.toPath();
                Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult preVisitDirectory(Path p, BasicFileAttributes a) throws IOException {
                        p.toFile().deleteOnExit();
                        return a.isSymbolicLink() ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult visitFile(Path p, BasicFileAttributes a) throws IOException {
                        p.toFile().deleteOnExit();
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
        }
    }

    public static void moveFile(File src, File dst) throws IOException {
        if (src == null || dst == null || !src.exists() || src.isDirectory() || dst.exists()) {
            throw new IllegalArgumentException("Invalid input " + String.valueOf(src) + " or " + String.valueOf(dst));
        }
        if (!src.renameTo(dst)) {
            Files.move(src.toPath(), dst.toPath(), new CopyOption[0]);
        }
    }

    public static void moveDirectory(File src, File dst) throws IOException {
        if (src == null || dst == null || !src.exists() || !src.isDirectory() || dst.exists()) {
            throw new IllegalArgumentException("Invalid input " + String.valueOf(src) + " or " + String.valueOf(dst));
        }
        if (!src.renameTo(dst)) {
            Path from = src.toPath().toAbsolutePath().normalize();
            Path to = dst.toPath().toAbsolutePath().normalize();
            if (to.startsWith(from)) {
                throw new IllegalArgumentException("Cannot move directory to itself or its subdirectory");
            }
            JavaUtils.moveDirectory(from, to);
        }
    }

    private static void moveDirectory(Path src, Path dst) throws IOException {
        Files.createDirectories(dst, new FileAttribute[0]);
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(src);){
            for (Path from : stream) {
                Path to = dst.resolve(from.getFileName());
                if (Files.isDirectory(from, new LinkOption[0])) {
                    JavaUtils.moveDirectory(from, to);
                    continue;
                }
                Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
            }
        }
        Files.delete(src);
    }

    public static void copyDirectory(File src, File dst) throws IOException {
        if (src == null || dst == null || !src.exists() || !src.isDirectory() || dst.exists() && !dst.isDirectory()) {
            throw new IllegalArgumentException("Invalid input file " + String.valueOf(src) + " or directory " + String.valueOf(dst));
        }
        final Path from = src.toPath().toAbsolutePath().normalize();
        final Path to = dst.toPath().toAbsolutePath().normalize();
        if (to.startsWith(from)) {
            throw new IllegalArgumentException("Cannot copy directory to itself or its subdirectory");
        }
        Files.createDirectories(to, new FileAttribute[0]);
        Files.walkFileTree(from, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Files.createDirectories(to.resolve(from.relativize(dir)), new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.copy(file, to.resolve(from.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static int nonNegativeHash(Object obj) {
        if (obj == null) {
            return 0;
        }
        int hash = obj.hashCode();
        return hash != Integer.MIN_VALUE ? Math.abs(hash) : 0;
    }

    public static ByteBuffer stringToBytes(String s) {
        return ByteBuffer.wrap(s.getBytes(StandardCharsets.UTF_8));
    }

    public static String bytesToString(ByteBuffer b) {
        return StandardCharsets.UTF_8.decode(b.slice()).toString();
    }

    public static long sizeOf(File file) throws IOException {
        if (!file.exists()) {
            throw new IllegalArgumentException(file.getAbsolutePath() + " not found");
        }
        return JavaUtils.sizeOf(file.toPath());
    }

    public static long sizeOf(Path dirPath) throws IOException {
        final AtomicLong size = new AtomicLong(0L);
        Files.walkFileTree(dirPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                size.addAndGet(attrs.size());
                return FileVisitResult.CONTINUE;
            }
        });
        return size.get();
    }

    public static void cleanDirectory(File dir) throws IOException {
        if (dir == null || !dir.exists() || !dir.isDirectory()) {
            throw new IllegalArgumentException("Invalid input directory " + String.valueOf(dir));
        }
        JavaUtils.cleanDirectory(dir.toPath());
    }

    private static void cleanDirectory(final Path rootDir) throws IOException {
        Files.walkFileTree(rootDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                if (!dir.equals(rootDir)) {
                    Files.delete(dir);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void deleteRecursively(File file) throws IOException, InterruptedException {
        JavaUtils.deleteRecursively(file, null);
    }

    public static void deleteRecursively(File file, FilenameFilter filter) throws IOException, InterruptedException {
        if (file == null) {
            return;
        }
        if (!(!isUnix || filter != null || isMac && JavaUtils.isTesting())) {
            try {
                JavaUtils.deleteRecursivelyUsingUnixNative(file);
                return;
            }
            catch (IOException e) {
                logger.warn("Attempt to delete using native Unix OS command failed for path = {}. Falling back to Java IO way", e, MDC.of(LogKeys.PATH, file.getAbsolutePath()));
            }
        }
        JavaUtils.deleteRecursivelyUsingJavaIO(file, filter);
    }

    private static void deleteRecursivelyUsingJavaIO(File file, FilenameFilter filter) throws IOException, InterruptedException {
        boolean deleted;
        BasicFileAttributes fileAttributes = JavaUtils.readFileAttributes(file);
        if (fileAttributes == null || !file.exists() && !fileAttributes.isSymbolicLink()) {
            return;
        }
        if (fileAttributes.isDirectory()) {
            IOException savedIOException = null;
            for (File child : JavaUtils.listFilesSafely(file, filter)) {
                try {
                    JavaUtils.deleteRecursively(child, filter);
                }
                catch (IOException e) {
                    savedIOException = e;
                }
            }
            if (savedIOException != null) {
                throw savedIOException;
            }
        }
        if ((fileAttributes.isRegularFile() || fileAttributes.isSymbolicLink() || fileAttributes.isDirectory() && JavaUtils.listFilesSafely(file, null).length == 0) && !(deleted = file.delete()) && file.exists()) {
            throw new IOException("Failed to delete: " + file.getAbsolutePath());
        }
    }

    private static BasicFileAttributes readFileAttributes(File file) {
        try {
            return Files.readAttributes(file.toPath(), BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
        }
        catch (IOException e) {
            return null;
        }
    }

    private static void deleteRecursivelyUsingUnixNative(File file) throws InterruptedException, IOException {
        ProcessBuilder builder = new ProcessBuilder("rm", "-rf", file.getAbsolutePath());
        Process process = null;
        int exitCode = -1;
        try {
            builder.redirectErrorStream(true);
            builder.redirectOutput(new File("/dev/null"));
            process = builder.start();
            exitCode = process.waitFor();
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException("Failed to delete: " + file.getAbsolutePath(), e);
        }
        finally {
            if (process != null) {
                process.destroy();
            }
        }
        if (exitCode != 0 || file.exists()) {
            throw new IOException("Failed to delete: " + file.getAbsolutePath());
        }
    }

    private static File[] listFilesSafely(File file, FilenameFilter filter) throws IOException {
        if (file.exists()) {
            File[] files = file.listFiles(filter);
            if (files == null) {
                throw new IOException("Failed to list files for dir: " + String.valueOf(file));
            }
            return files;
        }
        return new File[0];
    }

    public static Set<Path> listPaths(File dir) throws IOException {
        if (dir == null) {
            throw new IllegalArgumentException("Input directory is null");
        }
        if (!dir.exists() || !dir.isDirectory()) {
            return Set.of();
        }
        try (Stream<Path> stream = Files.walk(dir.toPath(), FileVisitOption.FOLLOW_LINKS);){
            Set set = stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).collect(Collectors.toCollection(HashSet::new));
            return set;
        }
    }

    public static Set<File> listFiles(File dir) throws IOException {
        if (dir == null) {
            throw new IllegalArgumentException("Input directory is null");
        }
        if (!dir.exists() || !dir.isDirectory()) {
            return Set.of();
        }
        try (Stream<Path> stream = Files.walk(dir.toPath(), FileVisitOption.FOLLOW_LINKS);){
            Set set = stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(Path::toFile).collect(Collectors.toCollection(HashSet::new));
            return set;
        }
    }

    public static long timeStringAs(String str, TimeUnit unit) {
        String lower = str.toLowerCase(Locale.ROOT).trim();
        try {
            Matcher m = TIME_STRING_PATTERN.matcher(lower);
            if (!m.matches()) {
                throw new NumberFormatException("Failed to parse time string: " + str);
            }
            long val = Long.parseLong(m.group(1));
            String suffix = m.group(2);
            if (suffix != null && !timeSuffixes.containsKey(suffix)) {
                throw new NumberFormatException("Invalid suffix: \"" + suffix + "\"");
            }
            return unit.convert(val, suffix != null ? timeSuffixes.get(suffix) : unit);
        }
        catch (NumberFormatException e) {
            String timeError = "Time must be specified as seconds (s), milliseconds (ms), microseconds (us), minutes (m or min), hour (h), or day (d). E.g. 50s, 100ms, or 250us.";
            throw new NumberFormatException(timeError + "\n" + e.getMessage());
        }
    }

    public static long timeStringAsMs(String str) {
        return JavaUtils.timeStringAs(str, TimeUnit.MILLISECONDS);
    }

    public static long timeStringAsSec(String str) {
        return JavaUtils.timeStringAs(str, TimeUnit.SECONDS);
    }

    public static long byteStringAs(String str, ByteUnit unit) {
        String lower = str.toLowerCase(Locale.ROOT).trim();
        try {
            Matcher m = BYTE_STRING_PATTERN.matcher(lower);
            Matcher fractionMatcher = BYTE_STRING_FRACTION_PATTERN.matcher(lower);
            if (m.matches()) {
                long val = Long.parseLong(m.group(1));
                String suffix = m.group(2);
                if (suffix != null && !byteSuffixes.containsKey(suffix)) {
                    throw new NumberFormatException("Invalid suffix: \"" + suffix + "\"");
                }
                return unit.convertFrom(val, suffix != null ? byteSuffixes.get(suffix) : unit);
            }
            if (fractionMatcher.matches()) {
                throw new NumberFormatException("Fractional values are not supported. Input was: " + fractionMatcher.group(1));
            }
            throw new NumberFormatException("Failed to parse byte string: " + str);
        }
        catch (NumberFormatException e) {
            String byteError = "Size must be specified as bytes (b), kibibytes (k), mebibytes (m), gibibytes (g), tebibytes (t), or pebibytes(p). E.g. 50b, 100k, or 250m.";
            throw new NumberFormatException(byteError + "\n" + e.getMessage());
        }
    }

    public static long byteStringAsBytes(String str) {
        return JavaUtils.byteStringAs(str, ByteUnit.BYTE);
    }

    public static long byteStringAsKb(String str) {
        return JavaUtils.byteStringAs(str, ByteUnit.KiB);
    }

    public static long byteStringAsMb(String str) {
        return JavaUtils.byteStringAs(str, ByteUnit.MiB);
    }

    public static long byteStringAsGb(String str) {
        return JavaUtils.byteStringAs(str, ByteUnit.GiB);
    }

    public static byte[] bufferToArray(ByteBuffer buffer) {
        if (buffer.hasArray() && buffer.arrayOffset() == 0 && buffer.array().length == buffer.remaining()) {
            return buffer.array();
        }
        byte[] bytes = new byte[buffer.remaining()];
        buffer.get(bytes);
        return bytes;
    }

    public static File createDirectory(String root) throws IOException {
        return JavaUtils.createDirectory(root, "spark");
    }

    public static File createDirectory(String root, String namePrefix) throws IOException {
        if (namePrefix == null) {
            namePrefix = "spark";
        }
        int attempts = 0;
        int maxAttempts = 10;
        File dir = null;
        while (dir == null) {
            if (++attempts > maxAttempts) {
                throw new IOException("Failed to create a temp directory (under " + root + ") after " + maxAttempts + " attempts!");
            }
            try {
                dir = new File(root, namePrefix + "-" + String.valueOf(UUID.randomUUID()));
                Files.createDirectories(dir.toPath(), new FileAttribute[0]);
            }
            catch (IOException | SecurityException e) {
                logger.error("Failed to create directory {}", e, MDC.of(LogKeys.PATH, dir));
                dir = null;
            }
        }
        return dir.getCanonicalFile();
    }

    public static void readFully(ReadableByteChannel channel, ByteBuffer dst) throws IOException {
        int expected = dst.remaining();
        while (dst.hasRemaining()) {
            if (channel.read(dst) >= 0) continue;
            throw new EOFException(String.format("Not enough bytes in channel (expected %d).", expected));
        }
    }

    public static void readFully(InputStream in, byte[] arr, int off, int len) throws IOException {
        if (in == null || len < 0 || off < 0 || off > arr.length - len) {
            throw new IllegalArgumentException("Invalid input argument");
        }
        if (len != in.readNBytes(arr, off, len)) {
            throw new EOFException("Fail to read " + len + " bytes.");
        }
    }

    public static void copyURLToFile(URL url, File file) throws IOException {
        if (url == null || file == null || file.exists() && file.isDirectory()) {
            throw new IllegalArgumentException("Invalid input " + String.valueOf(url) + " or " + String.valueOf(file));
        }
        Files.createDirectories(file.getParentFile().toPath(), new FileAttribute[0]);
        try (InputStream in = url.openStream();){
            Files.copy(in, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static String join(List<Object> arr, String sep) {
        if (arr == null) {
            return "";
        }
        StringJoiner joiner = new StringJoiner(sep == null ? "" : sep);
        for (Object a : arr) {
            joiner.add(a == null ? "" : a.toString());
        }
        return joiner.toString();
    }

    public static String stackTraceToString(Throwable t) {
        if (t == null) {
            return "";
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try (PrintWriter writer = new PrintWriter(out);){
            t.printStackTrace(writer);
            writer.flush();
        }
        return out.toString(StandardCharsets.UTF_8);
    }

    public static int checkedCast(long value) {
        if (value > Integer.MAX_VALUE || value < Integer.MIN_VALUE) {
            throw new IllegalArgumentException("Cannot cast to integer.");
        }
        return (int)value;
    }

    public static boolean contentEquals(File file1, File file2) throws IOException {
        Path path2;
        if (file1 == null && file2 != null || file1 != null && file2 == null) {
            return false;
        }
        if (file1 == null && file2 == null || !file1.exists() && !file2.exists()) {
            return true;
        }
        if (!file1.exists() || !file2.exists()) {
            return false;
        }
        if (file1.isDirectory() || file2.isDirectory()) {
            throw new IllegalArgumentException("Input is not a file: %s or %s".formatted(file1, file2));
        }
        if (file1.length() != file2.length()) {
            return false;
        }
        Path path1 = file1.toPath();
        return Files.isSameFile(path1, path2 = file2.toPath()) || Files.mismatch(path1, path2) == -1L;
    }

    public static String toString(InputStream in) throws IOException {
        return new String(in.readAllBytes(), StandardCharsets.UTF_8);
    }

    public static boolean isTesting() {
        return System.getenv("SPARK_TESTING") != null || System.getProperty("spark.testing") != null;
    }

    public static void checkArgument(boolean check, String msg, Object ... args) {
        if (!check) {
            throw new IllegalArgumentException(String.format(msg, args));
        }
    }

    public static void checkState(boolean check, String msg, Object ... args) {
        if (!check) {
            throw new IllegalStateException(String.format(msg, args));
        }
    }
}

